#ifndef _RASTER_FONT_H_
#define _RASTER_FONT_H_

#include <sys/types.h>
#include <sys/cdefs.h>

/****************************************************************************************************************/

struct fontmetrics {
  uint fm_maxwidth;			// Maximum character width used
  uint fm_minwidth;			// Minimum character width used (Width of a space)
  uint fm_height;			// Height of character in pixels (ascent + descent + 1)
  uint fm_first_char;			// First character present in glyph map
  uint fm_last_char;			// Last character present in glyph map
  uint fm_ascent;			// Distance from baseline to top of font
  uint fm_descent;			// Distance from baseline to bottom of font
  const char *fm_name;			// Name of font (Could be NULL)
};

/****************************************************************************************************************/
//
// Properties of a specific glyph.
//
// If any of the bearing values is negative, the glyph overlaps with the adjacent glyph.
//
struct glyphmetrics {
  int  gm_left_bearing;			// Number of empty columns to the left of the glyph
  uint gm_width;			// Number of pixels that X advances after a glyph has been rendered
  int  gm_right_bearing;		// Number of empty columns to the right of the glyph
};

/****************************************************************************************************************/
//
// Initialization
//
struct font;

__BEGIN_DECLS
struct font *font_load(const char *filename);
struct font *font_init(const void *data, size_t bytes);
void font_free(struct font *font);
__END_DECLS

/****************************************************************************************************************/
//
// Multiple font support
//
__BEGIN_DECLS
void font_insert_after(struct font *font, struct font *insert);
void font_insert_before(struct font *font, struct font *insert);
void font_remove(struct font *font);
struct font *font_next(struct font *font);
void font_free_list(struct font *font);
__END_DECLS

/****************************************************************************************************************/
//
// Private data field
//
__BEGIN_DECLS
void font_set_data(struct font *font, void *data);
void *font_get_data(struct font *font);
__END_DECLS

/****************************************************************************************************************/
//
// Properties
//
__BEGIN_DECLS

// Information about all glyphs
const struct fontmetrics *font_metrics(struct font *font);

// Fills in information about a specific glyph (Returns -1 on error, 0 on success)
int font_glyph_metrics(struct font *font, uint ch, struct glyphmetrics *gm);

// Returns width of given character in pixels
uint font_char_width(struct font *font, uint ch);

// Returns width of a string in pixels
uint font_text_width(struct font *font, const char *str);
uint font_text_width_utf8(struct font *font, const char *str);

uint font_text_width_len(struct font *font, const char *str, size_t len);
uint font_text_width_len_utf8(struct font *font, const char *str, size_t len);

// Quick access to font metrics
uint font_height(struct font *font);
uint font_ascent(struct font *font);
uint font_descent(struct font *font);

__END_DECLS

/****************************************************************************************************************/
//
// Debug
//
__BEGIN_DECLS
void font_dump(struct font *font);
__END_DECLS

/****************************************************************************************************************/
//
// Rendering
//
__BEGIN_DECLS
uint font_putc(struct font *font, struct pixmap *pm, int x, int y, uint ch);

uint font_puts(struct font *font, struct pixmap *pm, int x, int y, const char *str);
uint font_puts_utf8(struct font *font, struct pixmap *pm, int x, int y, const char *str);

void font_puts_centered(struct font *font, struct pixmap *pm, int y, const char *str);
void font_puts_centered_utf8(struct font *font, struct pixmap *pm, int y, const char *str);
__END_DECLS

/****************************************************************************************************************/
//
// Box model rendering
//
struct font_render {
  struct font *fr_font;
  struct pixmap *fr_pm;
  int  fr_x;
  int  fr_y;
  uint fr_w;
  uint fr_h;
  uint fr_align;
};

#define FONT_ALIGN_LEFT		0 /* Also defined in lua/gfx.lh */
#define FONT_ALIGN_RIGHT	1
#define FONT_ALIGN_CENTER	2
#define FONT_ALIGN_BLOCKCENTER	3
#define  FONT_ALIGN_HORIZONTAL	3

#define FONT_ALIGN_BASELINE	0
#define FONT_ALIGN_TOP		4
#define FONT_ALIGN_BOTTOM	8
#define FONT_ALIGN_VCENTER	12
#define  FONT_ALIGN_VERTICAL	12

#define FONT_ALIGN_WORDWRAP	16
#define FONT_ALIGN_JUSTIFY	32

#define FONT_LTR		64
#define FONT_RTL		128

#define FONT_ARABIC_JOINING	256
#define FONT_ARABIC_NUMBERS	512

#define FONT_RENDER_PAINT	0x20000000	// If not set, does not actually paint. Useful together with MEASURE.
#define FONT_RENDER_MEASURE	0x40000000	// Places measured size in fr_x, fr_y, fr_w, fr_h

#define FONT_RENDER_DEBUG	0x80000000

void font_render_utf8(struct font_render *fr, const char *text, uint textlen);

/****************************************************************************************************************/
#endif

