/**
 * \file ssl_cache.h
 *
 * \brief SSL session cache implementation
 *
 *  Copyright (C) 2006-2013, Brainspark B.V.
 *
 *  This file is part of PolarSSL (http://www.polarssl.org)
 *  Lead Maintainer: Paul Bakker <polarssl_maintainer at polarssl.org>
 *
 *  All rights reserved.
 *
 *  This code is re-licensed to Teratronik elektronische systeme gmbh (Licensee)
 *  by Offspark B.V. under a Commercial License for use in
 *  Teratronik v4 Controllers.
 *  
 *  Full terms of the Software License are stated in the relevant PolarSSL
 *  Commercial License Agreement (PL-TERATRONIK-130108).
 *  
 *  In short Licensee has a nonexclusive, perpetual, irrevocable worldwide
 *  license to use, make, have made, copy, modify and create derivative works
 *  of this Software in source and object code for purposes of developing and
 *  integrating the Software into the product(s) specified in the License.
 */
#ifndef POLARSSL_SSL_CACHE_H
#define POLARSSL_SSL_CACHE_H

#include "ssl.h"

#if defined(POLARSSL_THREADING_C)
#include "threading.h"
#endif

#if !defined(POLARSSL_CONFIG_OPTIONS)
#define SSL_CACHE_DEFAULT_TIMEOUT       86400   /*!< 1 day  */
#define SSL_CACHE_DEFAULT_MAX_ENTRIES      50   /*!< Maximum entries in cache */
#endif /* !POLARSSL_CONFIG_OPTIONS */

#ifdef __cplusplus
extern "C" {
#endif

typedef struct _ssl_cache_context ssl_cache_context;
typedef struct _ssl_cache_entry ssl_cache_entry;

/**
 * \brief   This structure is used for storing cache entries
 */
struct _ssl_cache_entry
{
#if defined(POLARSSL_HAVE_TIME)
    time_t timestamp;           /*!< entry timestamp    */
#endif
    ssl_session session;        /*!< entry session      */
#if defined(POLARSSL_X509_CRT_PARSE_C)
    x509_buf peer_cert;         /*!< entry peer_cert    */
#endif
    ssl_cache_entry *next;      /*!< chain pointer      */
};

/**
 * \brief Cache context
 */
struct _ssl_cache_context
{
    ssl_cache_entry *chain;     /*!< start of the chain     */
    int timeout;                /*!< cache entry timeout    */
    int max_entries;            /*!< maximum entries        */
#if defined(POLARSSL_THREADING_C)
    threading_mutex_t mutex;    /*!< mutex                  */
#endif
};

/**
 * \brief          Initialize an SSL cache context
 *
 * \param cache    SSL cache context
 */
void ssl_cache_init( ssl_cache_context *cache );

/**
 * \brief          Cache get callback implementation
 *                 (Thread-safe if POLARSSL_THREADING_C is enabled)
 *
 * \param data     SSL cache context
 * \param session  session to retrieve entry for
 */
int ssl_cache_get( void *data, ssl_session *session );

/**
 * \brief          Cache set callback implementation
 *                 (Thread-safe if POLARSSL_THREADING_C is enabled)
 *
 * \param data     SSL cache context
 * \param session  session to store entry for
 */
int ssl_cache_set( void *data, const ssl_session *session );

#if defined(POLARSSL_HAVE_TIME)
/**
 * \brief          Set the cache timeout
 *                 (Default: SSL_CACHE_DEFAULT_TIMEOUT (1 day))
 *
 *                 A timeout of 0 indicates no timeout.
 *
 * \param cache    SSL cache context
 * \param timeout  cache entry timeout
 */
void ssl_cache_set_timeout( ssl_cache_context *cache, int timeout );
#endif /* POLARSSL_HAVE_TIME */

/**
 * \brief          Set the cache timeout
 *                 (Default: SSL_CACHE_DEFAULT_MAX_ENTRIES (50))
 *
 * \param cache    SSL cache context
 * \param max      cache entry maximum
 */
void ssl_cache_set_max_entries( ssl_cache_context *cache, int max );

/**
 * \brief          Free referenced items in a cache context and clear memory
 *
 * \param cache    SSL cache context
 */
void ssl_cache_free( ssl_cache_context *cache );

#ifdef __cplusplus
}
#endif

#endif /* ssl_cache.h */
