/**
 * \file debug.h
 *
 * \brief Debug functions
 *
 *  Copyright (C) 2006-2013, Brainspark B.V.
 *
 *  This file is part of PolarSSL (http://www.polarssl.org)
 *  Lead Maintainer: Paul Bakker <polarssl_maintainer at polarssl.org>
 *
 *  All rights reserved.
 *
 *  This code is re-licensed to Teratronik elektronische systeme gmbh (Licensee)
 *  by Offspark B.V. under a Commercial License for use in
 *  Teratronik v4 Controllers.
 *  
 *  Full terms of the Software License are stated in the relevant PolarSSL
 *  Commercial License Agreement (PL-TERATRONIK-130108).
 *  
 *  In short Licensee has a nonexclusive, perpetual, irrevocable worldwide
 *  license to use, make, have made, copy, modify and create derivative works
 *  of this Software in source and object code for purposes of developing and
 *  integrating the Software into the product(s) specified in the License.
 */
#ifndef POLARSSL_DEBUG_H
#define POLARSSL_DEBUG_H

#include "config.h"
#include "ssl.h"
#if defined(POLARSSL_ECP_C)
#include "ecp.h"
#endif

#if defined(POLARSSL_DEBUG_C)

#define SSL_DEBUG_MSG( level, args )                    \
    debug_print_msg( ssl, level, __FILE__, __LINE__, debug_fmt args );

#define SSL_DEBUG_RET( level, text, ret )                \
    debug_print_ret( ssl, level, __FILE__, __LINE__, text, ret );

#define SSL_DEBUG_BUF( level, text, buf, len )           \
    debug_print_buf( ssl, level, __FILE__, __LINE__, text, buf, len );

#if defined(POLARSSL_BIGNUM_C)
#define SSL_DEBUG_MPI( level, text, X )                  \
    debug_print_mpi( ssl, level, __FILE__, __LINE__, text, X );
#endif

#if defined(POLARSSL_ECP_C)
#define SSL_DEBUG_ECP( level, text, X )                  \
    debug_print_ecp( ssl, level, __FILE__, __LINE__, text, X );
#endif

#if defined(POLARSSL_X509_CRT_PARSE_C)
#define SSL_DEBUG_CRT( level, text, crt )                \
    debug_print_crt( ssl, level, __FILE__, __LINE__, text, crt );
#endif

#else

#define SSL_DEBUG_MSG( level, args )            do { } while( 0 )
#define SSL_DEBUG_RET( level, text, ret )       do { } while( 0 )
#define SSL_DEBUG_BUF( level, text, buf, len )  do { } while( 0 )
#define SSL_DEBUG_MPI( level, text, X )         do { } while( 0 )
#define SSL_DEBUG_ECP( level, text, X )         do { } while( 0 )
#define SSL_DEBUG_CRT( level, text, crt )       do { } while( 0 )

#endif

#ifdef __cplusplus
extern "C" {
#endif

char *debug_fmt( const char *format, ... );

void debug_print_msg( const ssl_context *ssl, int level,
                      const char *file, int line, const char *text );

void debug_print_ret( const ssl_context *ssl, int level,
                      const char *file, int line,
                      const char *text, int ret );

void debug_print_buf( const ssl_context *ssl, int level,
                      const char *file, int line, const char *text,
                      unsigned char *buf, size_t len );

#if defined(POLARSSL_BIGNUM_C)
void debug_print_mpi( const ssl_context *ssl, int level,
                      const char *file, int line,
                      const char *text, const mpi *X );
#endif

#if defined(POLARSSL_ECP_C)
void debug_print_ecp( const ssl_context *ssl, int level,
                      const char *file, int line,
                      const char *text, const ecp_point *X );
#endif

#if defined(POLARSSL_X509_CRT_PARSE_C)
void debug_print_crt( const ssl_context *ssl, int level,
                      const char *file, int line,
                      const char *text, const x509_crt *crt );
#endif

#ifdef __cplusplus
}
#endif

#endif /* debug.h */
