#ifndef _RASTER_FONT_H_
#define _RASTER_FONT_H_

#include <sys/types.h>
#include <sys/cdefs.h>

/****************************************************************************************************************/

struct fontmetrics {
  uint fm_maxwidth;			// Maximum character width used
  uint fm_minwidth;			// Minimum character width used (Width of a space)
  uint fm_height;			// Height of character in pixels (ascent + descent + 1)
  uint fm_first_char;			// First character present in glyph map
  uint fm_last_char;			// Last character present in glyph map
  uint fm_ascent;			// Distance from baseline to top of font
  uint fm_descent;			// Distance from baseline to bottom of font
  const char *fm_name;			// Name of font (Could be NULL)
};

/****************************************************************************************************************/
//
// Properties of a specific glyph.
//
// If any of the bearing values is negative, the glyph overlaps with the adjacent glyph.
//
struct glyphmetrics {
  int  gm_left_bearing;			// Number of empty columns to the left of the glyph
  uint gm_width;			// Number of pixels that X advances after a glyph has been rendered
  int  gm_right_bearing;		// Number of empty columns to the right of the glyph
};

/****************************************************************************************************************/
//
// Initialization
//
struct font;

__BEGIN_DECLS
struct font *font_load(const char *filename);
struct font *font_init(const void *data, size_t bytes);
void font_free(struct font *font);
__END_DECLS

/****************************************************************************************************************/
//
// Properties
//
__BEGIN_DECLS

// Information about all glyphs
const struct fontmetrics *font_metrics(struct font *font);

// Fills in information about a specific glyph (Returns -1 on error, 0 on success)
int font_glyph_metrics(struct font *font, uint ch, struct glyphmetrics *gm);

// Returns width of given character in pixels
uint font_char_width(struct font *font, uint ch);

// Returns width of a string in pixels
uint font_text_width(struct font *font, const char *str);
uint font_text_width_utf8(struct font *font, const char *str);

uint font_text_width_len(struct font *font, const char *str, size_t len);
uint font_text_width_len_utf8(struct font *font, const char *str, size_t len);

// Quick access to font metrics
uint font_height(struct font *font);
uint font_ascent(struct font *font);
uint font_descent(struct font *font);

__END_DECLS

/****************************************************************************************************************/
//
// Debug
//
__BEGIN_DECLS
void font_dump(struct font *font);
__END_DECLS

/****************************************************************************************************************/
//
// Rendering
//
__BEGIN_DECLS
uint font_putc(struct font *font, struct pixmap *pm, int x, int y, uint ch);

uint font_puts(struct font *font, struct pixmap *pm, int x, int y, const char *str);
uint font_puts_utf8(struct font *font, struct pixmap *pm, int x, int y, const char *str);

void font_puts_centered(struct font *font, struct pixmap *pm, int y, const char *str);
void font_puts_centered_utf8(struct font *font, struct pixmap *pm, int y, const char *str);

int font_putc_undef(struct pixmap *pm, int xx, int yy, int ww, int hh, uint ch);
__END_DECLS

/****************************************************************************************************************/
#endif

